package ru.autosome.di;

import ru.autosome.ChIPApp;
import ru.autosome.assist.Conductor;
import ru.autosome.assist.Occurrence;
import ru.autosome.di.ytilib.MunkResult;
import ru.autosome.di.ytilib.Sequence;
import ru.autosome.di.ytilib.WPCM;

import java.util.ArrayList;
import java.util.List;

public class ChIPHorde extends ChIPApp {

  private int totalSeqs;
  private List<MunkResult> results = new ArrayList<MunkResult>();

  private List<Integer[]> lengthRanges;
  private ChIPAct.PreprocessMode mode;
  private String[] params;
  private List<Sequence[]> sequenceSets;
  private ChIPAct.Parameters actParameters;
  private ChIPMunk.Parameters munkParameters;

  public static void main(String[] args) {
    ChIPHorde chiphorde = new ChIPHorde(args, Conductor.defaultConductor);
    chiphorde.launchViaConductor();
  }

  public ChIPHorde(List<Integer[]> lengthRanges, ChIPAct.PreprocessMode mode,
                   ChIPAct.Parameters actParameters, ChIPMunk.Parameters munkParameters) {
    super(actParameters.getConductor());

    this.lengthRanges = lengthRanges;
    int sum = 0;
    for (Integer[] lengthRange : lengthRanges) {
      sum += Math.abs(lengthRange[0] - lengthRange[1]) + 1;
    }
    conductor.setTotalTicks(sum * 9);

    this.mode = mode;
    this.verbose = munkParameters.getVerbose();

    actParameters.setMotifLength(1);

    ChIPAct falseAct = new ChIPAct(actParameters);
    this.sequenceSets = falseAct.sequenceSets;
    this.totalSeqs = falseAct.joinedSets.length;

    this.actParameters = actParameters;
    actParameters.setPreprocessList(results);
    actParameters.setPreprocessMode(mode);

    this.munkParameters = munkParameters;
  }

  public ChIPHorde(String[] args, Conductor conductor) {
    super(conductor);

    conductor.message("ru.autosome.di.ChIPHorde engine usage: <start_length1:stop_length1,start_length2:stop_length2..start_lengthN:stop_lengthN> <mode=(m)ask|(f)ilter|(d)ummy> <verbose=(n)o|(p)artial|(y)es> <ru.autosome.di.ChIPMunk parameters>");
    if (args.length < 5) throw new RuntimeException("not enough parameters");
    if (args[3].startsWith("o")) throw new RuntimeException("ChIPHorde cannot be used along with one-occurence-per-sequence (OOPS) mode");

    params = new String[args.length];
    System.arraycopy(args, 2, params, 2, params.length-2);

    char vchar = args[2].toUpperCase().charAt(0);
    if (vchar == 'Y') verbose = true;
    else if (vchar == 'N') verbose = false;
    else verbose = null;
    if (verbose != null && verbose)
      params[2] = "yes";
    else
      params[2] = "no";

    String[] lengthDiaps = args[0].split(",");
    lengthRanges = new ArrayList<Integer[]>();

    int sum = 0;
    for (int i = 0; i < lengthDiaps.length; i++) {
      int startLength = new Integer(lengthDiaps[i].split(":")[0]);
      int stopLength = new Integer(lengthDiaps[i].split(":")[1]);
      lengthRanges.add(new Integer[] { startLength, stopLength });
      sum += Math.abs(startLength-stopLength)+1;
    }
    conductor.setTotalTicks(sum * 9);

    char modeChar = args[1].toUpperCase().charAt(0);
    mode = modeChar == 'F' ? PreprocessMode.FILTER : (modeChar == 'M' ? PreprocessMode.MASK : (modeChar == 'D' ? PreprocessMode.DUMMY : null) );
    if (mode == null) throw new RuntimeException("unknown preprocessing mode " + args[1]);

    String[] newparams = new String[args.length-3];
    System.arraycopy(args, 4, newparams, 1, args.length-4);
    newparams[0] = "1";

    ChIPAct falseAct = new ChIPAct(newparams, conductor);

    sequenceSets = falseAct.sequenceSets;
    totalSeqs = falseAct.joinedSets.length;

  }

  public List<MunkResult> getResult() {
    return results;
  }

  public Object launch() {
    if (verbose != null && !verbose) conductor.silence();

    String message = null;
    int sumticks = 0;
    for (int i = 0; i < lengthRanges.size(); i++) {
      int startLength = lengthRanges.get(i)[0];
      int stopLength = lengthRanges.get(i)[1];

      conductor.message("starting " + i + "-th iteration, " + mode.toString() + " mode");

      ChIPMunk chipmunk;
      if (params != null) {
        params[0] = "" + startLength;
        params[1] = "" + stopLength;
        chipmunk = new ChIPMunk(params, conductor, mode, results);
      } else {
        munkParameters.setStartLength(startLength);
        munkParameters.setStopLength(stopLength);
        chipmunk = new ChIPMunk(actParameters, munkParameters);
      }

      MunkResult munkResult = (MunkResult)chipmunk.launchViaConductor();
      if (munkResult == null) {
        message = "got no result from the last stage, please check the program log";
        break;
      }

      WPCM motif = munkResult.getWPCM();

      sumticks += (Math.abs(startLength-stopLength)+1)*9;
      conductor.updateTicks(sumticks);

      if (motif == null) {
        message = "empty motif came from the step " + i + ", unknown error";
        break;
      }

      results.add(munkResult);

    }

    conductor.verbose();

    conductor.output("OUTC", "ru.autosome.di.ChIPHorde");
    conductor.output("INFO", "found " + results.size() + " motifs, used " + totalSeqs + " sequences");

    // dump all collected motifs and report final motif count
    for (int i = 0; i < results.size(); i++) {

      MunkResult result = results.get(i);

      conductor.output("MOTF", "motif # " + i);
      result.printout(conductor, verbose == null || verbose);
      conductor.output("LENG", result.getWPCM().length());
      conductor.output("KDDC", result.getWPCM().kdidic(results.get(i).getBackground()));
      ChIPAct.prettyOutputWPCM(conductor, result.getWPCM().getMatrix());
      conductor.output("THRE", result.getThreshold());
      // make mono WPCM -> consensus
      conductor.output("IUPA", new ru.autosome.ytilib.WPCM(result.getWPCM()).consensus1());

      ChIPAct.prettyOutputPWM(conductor, result.getPWM().getMatrix());
      ChIPAct.prettyOutputBackground(conductor, result.getBackground());
      conductor.output("PVAL", result.pvalue());
      conductor.output("DIAG", result.getDiagnosis());

      if (verbose != null && verbose) {
        // report motif occurrences for each sequence set for each sequence using default thresholds
        reportOccurrences(conductor, result.getPWM(), result.getThreshold());
      }

    }

    conductor.output("DIAG", message != null ? "fail; " + message : "success; found a given number of motifs");
    conductor.setStatus(message != null ? Conductor.Status.FAIL : Conductor.Status.SUCCESS);

    return results;
  }

  public void reportOccurrences(Conductor conductor, WPCM pm, double threshold) {
    // using sequenceSets to gather and report all hits
    for (int i = 0; i < sequenceSets.size(); i++) {
      reportOccurrences(conductor, pm, threshold, sequenceSets.get(i), new StringBuilder().append(i));
    }

  }

  private static void reportOccurrences(Conductor conductor, WPCM pm, double threshold, Sequence[] seqs, StringBuilder prefix) {

    for (int i = 0; i < seqs.length; i++) {
      StringBuilder printout = new StringBuilder(prefix);
      printout.append(";").append(i).append(";");

      Sequence seq = seqs[i];
      List<Occurrence> occurrences = seq.gatherOccurrences(pm, threshold);
      for (Occurrence occurrence : occurrences) {
        printout.append(" ").append(occurrence);
      }

      if (occurrences.size() > 0) {
        conductor.output("OCCS", printout);
      }
    }

  }

}
