#!/usr/bin/ruby
app_dir = File.dirname(File.expand_path(__FILE__))
$: << "#{app_dir}/ytilib"

require "ytilib.rb"

report "pmflogo3.rb started, usage: <in_file_name|IUPAC> <out_file_name> [<words_count>=default] [<x_unit>=100] [<y_unit>=200] [<icd_mode>=discrete|weblogo] [<revcomp>=no|yes] [<scheme>=nucl_simpa] [<paper_mode>=no|yes] [<threshold_lines>=yes|no]"
start __FILE__
exit(2) if ARGV.size < 2

in_file_name, out_file_name = ARGV[0], ARGV[1]
words_count = ARGV[2] && ARGV[2] != "default" ? ARGV[2].to_f : nil
x_unit = ARGV[3] == nil || ARGV[3] == "default" ? 100 : ARGV[3].to_i
y_unit = ARGV[4] == nil || ARGV[4] == "default" ? 200 : ARGV[4].to_i

icd_mode = ARGV[5] == nil || ARGV[5] != "weblogo" ? :discrete : :weblogo
revcomp = ARGV[6] && ARGV[6] != "no" && ARGV[6] != "false" && ARGV[6] != "direct"
scheme = ARGV[7] == nil ? "nucl_simpa" : ARGV[7]
paper_mode = ARGV[8] && ARGV[8] != "no" && ARGV[8] != "false"
threshold_lines = !ARGV[9] || (ARGV[9] != "no" && ARGV[9] != "false")

scheme_dir = File.join(app_dir, scheme)
ext, pm = File.ext_wo_name(in_file_name), nil

x_size, y_size = nil, y_unit

case ext
  when "pat", "pwm", "pcm", "pfm"
    pm = PM.load(in_file_name)
    if pm.words_count
      pm.fixwc
      pm = pm.get_safe_ppm
    end
    pm.words_count = words_count if words_count
    
  when "mfa", "fasta", "plain", "txt", "words", "fa"
    pm = PM.new_pcm(Ytilib.read_seqs2array(in_file_name))
    pm = pm.get_ppm
    pm.words_count = words_count if words_count
  when "xml"
    pm = PM.from_bismark(Bismark.new(in_file_name).elements["//PPM"])
    pm.words_count = words_count if words_count
  when in_file_name
    pm = PPM.from_IUPAC(in_file_name.upcase)
end

#(0...pm.size).each { |i|
#  p pm.matrix['A'][i] + pm.matrix['C'][i] + pm.matrix['G'][i] + pm.matrix['T'][i]
#}

checkerr("bad input file") { pm == nil }
unless pm.words_count
  report "words count for PM is undefined, assuming weblogo mode"
  icd_mode = :weblogo
end

if icd_mode == :weblogo
  class PPM
    def get_logo!
      
      rseq = []
      @matrix['A'].each_index { |i|
        rseq << 2 + ['A','C','G','T'].inject(0) { |sum, l| 
          pn = @matrix[l][i] 
          sum += pn == 0 ? 0 : pn * Math.log(pn) / Math.log(2)
        }
      }
      
      @matrix['A'].each_index { |i|
        ['A','C','G','T'].each { |l| 
          @matrix[l][i] *= rseq[i] / 2 # so we can handle a '2 bit' scale here
        }
      }
      
      @matrix
    end
  end
else
  class PPM
    def get_logo!
      
      report "called"
      
      checkerr("words count is undefined") { !words_count }
      
      rseq = []
      
      @matrix['A'].each_index { |i|
        
        infoi = infocod(i)
        infoi = icd4of4 if infoi < icd4of4 # for highly unbalanced matrices!
        
        rseq << (icd4of4 == 0 ? 1.0 : ( (infoi - icd4of4) / icd4of4 ).abs)
      }
      
      @matrix['A'].each_index { |i|
        ['A','C','G','T'].each { |l| 
          @matrix[l][i] *= rseq[i]
        }
      }
      
      @matrix
    end
    
    def get_line(v)
      ( (v - icd4of4) / icd4of4 ).abs
    end
    
  end
end

if icd_mode == :discrete && pm.words_count
  line2of4 = y_size - pm.get_line(pm.icd2of4)*y_unit
  lineThc = y_size - pm.get_line(pm.icdThc)*y_unit
  lineTlc = y_size - pm.get_line(pm.icdTlc)*y_unit
end

pm.revcomp! if revcomp
pm = pm.get_logo!

x_size = x_unit * pm['A'].size
report "using x_unit=#{x_unit}, y_unit=#{y_unit}, scheme=#{scheme}"
report "using x_size=#{x_size}, y_size=#{y_size}"

require 'RMagick'

lp = File.exist?("#{scheme_dir}/a.png") ? {'A' => "#{scheme_dir}/a.png", 'C' => "#{scheme_dir}/c.png", 'G' => "#{scheme_dir}/g.png", 'T' => "#{scheme_dir}/t.png"} : {'A' => "#{scheme_dir}/a.gif", 'C' => "#{scheme_dir}/c.gif", 'G' => "#{scheme_dir}/g.gif", 'T' => "#{scheme_dir}/t.gif"}

i_letters = [lp['A'],lp['C'], lp['G'], lp['T']].collect { |path|
  img = Magick::Image::read(path).first
}

i_logo = Magick::ImageList.new

unless paper_mode
  
  if icd_mode == :discrete
    i_logo.new_image(x_size, y_size, Magick::HatchFill.new('transparent', 'transparent'))
    
    if threshold_lines
      dr = Magick::Draw.new
      dr.fill('transparent')
      
      dr.stroke_width(y_size / 200.0)
      dr.stroke_dasharray(7,7)
      
      dr.stroke('silver')
      dr.line(0, line2of4, x_size, line2of4)
      dr.line(0, lineThc, x_size, lineThc)
      dr.line(0, lineTlc, x_size, lineTlc)
      
      dr.draw(i_logo)
    end
    
  else
    i_logo.new_image(x_size, y_size, Magick::HatchFill.new('white', 'bisque'))
    
  end
else
  i_logo.new_image(x_size, y_size)
end

(0...pm['A'].size).each { |i|
  y_pos = 0
  sorted_letters = ['A', 'C', 'G', 'T'].collect { |letter| {:score => pm[letter][i], :letter => letter} }.sort_by { |pair| pair[:score] }.collect { |pair| pair[:letter] }.reverse  
  sorted_letters.each { |letter|
    next if y_unit * pm[letter][i] <= 1
    letter_index = {'A' => 0, 'C' => 1, 'G' => 2, 'T' => 3}[letter]
    y_block = (y_unit * pm[letter][i]).round
    
    my_clone = i_letters[letter_index].dup
    my_clone_channels = my_clone.separate(Magick::AllChannels)
    my_clone_channels.each { |channel|
      channel.resize!(x_unit, y_block)
    }
    my_clone = my_clone_channels.combine
    #my_clone.resize!(x_unit, y_block)

    i_logo << my_clone
    
    #i_logo.cur_image.alpha(Magick::ActivateAlphaChannel)
     #, Magick::CubicFilter)
    
    y_pos += y_block
    i_logo.cur_image.page = Magick::Rectangle.new(0, 0, i*x_unit, y_size - y_pos )
  }
}

i_logo = i_logo.flatten_images
i_logo.cur_image.border!(x_unit / 100 + 1, x_unit / 100 + 1, icd_mode == :discrete ? "green" : "red") if paper_mode

i_logo.write(out_file_name)
