package ru.autosome.di;

import ru.autosome.ChIPApp;
import ru.autosome.assist.AShapeProvider;
import ru.autosome.assist.Conductor;
import ru.autosome.assist.Occurrence;
import ru.autosome.di.engine.Task;
import ru.autosome.di.engine.DoptiStep;
import ru.autosome.di.ytilib.*;
import ru.autosome.di.ytilib.MunkResult;
import ru.autosome.di.ytilib.Peak;
import ru.autosome.di.ytilib.RNASequence;
import ru.autosome.di.ytilib.SeedCask;
import ru.autosome.di.ytilib.Sequence;
import ru.autosome.di.ytilib.WPCM;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;
import java.util.Comparator;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

import static ru.autosome.assist.Helpers.makeTriangleProfile;

public class ChIPAct extends ChIPApp {

  private int threadCount;
  private int iterationLimit;
  private int stepLimit;
  private int tryLimit;
  private SeedCask seeds;

  private WPCM best;
  private double bestInfocod;
  private Integer motifLength;

  protected List<Sequence[]> sequenceSets;
  protected Sequence[] joinedSets;
  protected double[] background = Sequence.uniformBackground;

  private WPCM result = null;
  private double[] shape;

  public WPCM getResult() {
    return result;
  }

  public synchronized WPCM getBest() {
    return new WPCM(best);
  }

  public synchronized double getBestInfocod() {
    return bestInfocod;
  }

  public synchronized void checkBest(WPCM pm) {
    double newicd = pm.kdidic(background);
    if (newicd > bestInfocod) {
      conductor.message("found new GMLA with higher KDIDIC " + newicd + " > " + bestInfocod);
      best = pm;
      bestInfocod = best.kdidic(background);
    }
  }

  public static void main(String[] args) {
    ChIPAct chipstep = new ChIPAct(args, Conductor.defaultConductor);
    chipstep.verbose = true;
    chipstep.launchViaConductor();
  }

  public ChIPAct(String[] args, Conductor conductor) {
    this(args, conductor, null, null);
  }

  public ChIPAct(String[] args, Conductor conductor, PreprocessMode preprocessMode, List<MunkResult> preprocessList) {
    super(conductor);

    conductor.setTotalTicks(9);

    conductor.message("ru.autosome.di.ChIPAct engine usage: <length> <s:simple_set> <o:ordered_set> <w:weighted_set> <p:peak_set> <m:summit_set> <r:rna_set> [<try_limit>=200] [<step_limit>=20>] [<iter_limit>=1] [<thread_count>=2] [<seeds_set>=random|filename.mfa] [<background>=uniform|auto/local] [<motif_shape>=flat|single|double] [<disable_log_weighting>]");

    if (args.length < 2) throw new RuntimeException("not enough parameters");

    int setsCount = 0;
    for (String arg : args) {
      if (arg.length() > 1 && arg.charAt(1) == ':') setsCount++;
    }

    if (setsCount == 0) throw new RuntimeException("no input dataset found");

    motifLength = new Integer(args[0]);
    if (motifLength < 1) throw new RuntimeException("too small motif length");

    threadCount = (args.length - setsCount) > 4 ? new Integer(args[setsCount + 4]) : 2;
    iterationLimit = (args.length - setsCount) > 3 ? new Integer(args[setsCount + 3]) : 1;
    stepLimit = (args.length - setsCount) > 2 ? new Integer(args[setsCount + 2]) : 20;
    tryLimit = (args.length - setsCount) > 1 ? new Integer(args[setsCount + 1]) : 200;

    boolean logWeighting = (args.length <= setsCount + 8);

    sequenceSets = new ArrayList<Sequence[]>();
    List<Peak> allPeaks = new ArrayList<Peak>();
    for (int i = 1; i <= setsCount; i++) {

      String argsi = args[i].toLowerCase();
      SetType setype = argsi.startsWith("o:") ?
          SetType.ORDERED : argsi.startsWith("w:") ?
          SetType.WEIGHTED : argsi.startsWith("p:") ?
          SetType.PEAK : argsi.startsWith("r:") ?
          SetType.RNA : argsi.startsWith("m:") ?
          SetType.MIDPOINT : argsi.startsWith("i:") ?
          SetType.RNAMIDPOINT : argsi.startsWith("s:") ?
          SetType.SIMPLE : SetType.UNKNOWN;

      Sequence[] nset = null;

      switch (setype) {
        case WEIGHTED:
          nset = loadWeighted(args[i].substring(2), motifLength+1);
          break;
        case ORDERED:
          nset = loadSequences(args[i].substring(2), motifLength+1);
          for (int j = 0; j < nset.length; j++) {
            nset[j].weight = 1.0 / (j + 1);
          }
          break;
        case SIMPLE:
          nset = loadSequences(args[i].substring(2), motifLength+1);
          break;
        case PEAK:
          nset = loadPeaks(args[i].substring(2), motifLength+1);
          for (Peak wp: (Peak[])nset) {
            wp.soften(motifLength);
            wp.normalize(logWeighting);
          }
          /*java.util.Arrays.sort((Peak[])nset, new Comparator<Peak>() {
            public int compare(Peak o1, Peak o2) {
              return ((Double)o2.weight).compareTo(o1.weight);
            }
          });*/
          // big bug lives here; it is very dangerous to sort peaks since the sequence indexes in the output become wrong!
          allPeaks.addAll( java.util.Arrays.asList((Peak[])nset) );
          break;
        case RNA:
          nset = loadRNA(args[i].substring(2), motifLength+1);
          break;
        case MIDPOINT:
          nset = loadMidpoint(args[i].substring(2), motifLength+1, false);
          for (Peak wp: (Peak[])nset) {
            wp.soften(motifLength);
          }
          allPeaks.addAll( java.util.Arrays.asList((Peak[])nset) );
          break;
        case RNAMIDPOINT:
          nset = loadMidpoint(args[i].substring(2), motifLength+1, true);
          for (Peak wp: (Peak[])nset) {
            wp.soften(motifLength);
          }
          allPeaks.addAll( java.util.Arrays.asList((Peak[])nset) );
          break;
        case UNKNOWN:
          throw new RuntimeException("unknown sequence set mode detected");
      }

      sequenceSets.add(nset);

    }

    if (args.length > setsCount + 7) {
      shape = args[setsCount+7].startsWith("s") ? AShapeProvider.SingleBox.shape(motifLength) : ( args[setsCount+7].startsWith("d") ? AShapeProvider.DoubleBox.shape(motifLength) : null );
    }

    makeJoinedSets(preprocessMode, preprocessList, logWeighting);
    if (joinedSets == null || joinedSets.length == 0) throw new RuntimeException("empty sequence set: everything is filtered out or sequences are shorter than the requested motif length");

    if (args.length > setsCount + 6) {
      if (args[setsCount + 6].equals("auto") || args[setsCount + 6].equals("local")) {

        background = Sequence.background(joinedSets);

      } else if (!args[setsCount+6].equals("uniform")) {
          throw new UnsupportedOperationException("unsupported background: " + args[setsCount+6]);
      }
    }

    conductor.message("gathering seeds...");
    seeds = new SeedCask(tryLimit, motifLength);
    if (args.length > setsCount + 5) {
      if (args[setsCount + 5].toLowerCase().equals("random")) {
        // do nothing
      } else if (args[setsCount + 5].toLowerCase().equals("peak")) {
        if (allPeaks.size() == 0) throw new RuntimeException("no peak data found");
        Peak[] peaks = allPeaks.toArray(new Peak[allPeaks.size()]);
        java.util.Arrays.sort(peaks, new Comparator<Peak>() {
          public int compare(Peak o1, Peak o2) {
            return ((Double)o2.weight).compareTo(o1.weight);
          }
        });
        seeds.fromPeaks(peaks);
      } else {
        seeds.fromSequences(loadSequences(args[setsCount+5], motifLength));
      }
    }
    conductor.message("gathered " + seeds.count() + " seeds");
    
  }

  private static Sequence[] loadMidpoint(String filename, Integer minLength, boolean rnaMode) {
    List<Peak> resa = new ArrayList<Peak>();

    try {
      BufferedReader bin = new BufferedReader(new InputStreamReader(new FileInputStream(filename)));
      String s;
      StringBuilder sb = new StringBuilder();
      int[] profile = null;

      while ((s = bin.readLine()) != null) {
        if (s.length() != 0 && s.charAt(0) == '>') {
          String[] pivis = s.substring(1).trim().split(" ");
          if (sb.length() >= minLength) {
            if (profile.length > 1) {
              throw new IOException("more than one middlepoint-summit position given in the fasta header");
            }

            double[] triangleProfile = makeTriangleProfile(sb.length(), profile[0]);
            resa.add(rnaMode ? new RNASequence(sb.toString(), triangleProfile): new Peak(sb.toString(), triangleProfile));
          }
          sb.setLength(0);
          profile = new int[pivis.length];
          for (int i = 0; i < pivis.length; i++) {
            profile[i] = Integer.parseInt(pivis[i]);
          }
          continue;
        }
        sb.append(s.trim());
      }
      if (sb.length() >= minLength) {

        if (profile.length > 1) {
          throw new IOException("more than one middlepoint-summit position given in the fasta header");
        }

        double[] triangleProfile = makeTriangleProfile(sb.length(), profile[0]);
        resa.add(rnaMode ? new RNASequence(sb.toString(), triangleProfile) : new Peak(sb.toString(), triangleProfile));
      }

    } catch (IOException e) {
      e.printStackTrace();
      throw new RuntimeException("unable to parse the sequences file " + filename);
    }

    Peak[] res = new Peak[resa.size()];

    return resa.toArray(res);
  }

  private static Sequence[] loadRNA(String filename, Integer minLength) {
    List<Sequence> resa = new ArrayList<Sequence>();

    try {
      BufferedReader bin = new BufferedReader(new InputStreamReader(new FileInputStream(filename)));
      String s;
      StringBuilder sb = new StringBuilder();

      while ((s = bin.readLine()) != null) {
        if (s.length() > 0 && s.charAt(0) == '>') {
          if (sb.length() >= minLength) {
            resa.add(new RNASequence(sb.toString()));
          }
          sb.setLength(0);
          continue;
        }
        sb.append(s);
      }
      if (sb.length() >= minLength) resa.add(new RNASequence(sb.toString()));

    } catch (IOException e) {
      throw new RuntimeException("unable to parse input sequences file " + filename);
    }

    Sequence[] res = new Sequence[resa.size()];
    return resa.toArray(res);
  }

  private void makeJoinedSets(PreprocessMode preprocessMode, List<MunkResult> preprocessList, boolean logWeighting) {

    int totalWeight = countSequences(sequenceSets);
    List<Sequence[]> preprocessedSets = null;

    if (preprocessMode != null && preprocessList == null) {
      throw new RuntimeException("preprocess mode is defined, but preprocess list is undefined; unknown error");
    }

    if (preprocessMode == null || preprocessList.size() == 0 || preprocessMode == PreprocessMode.DUMMY) {
      preprocessedSets = sequenceSets;
    } else if (preprocessMode == PreprocessMode.MASK) {
      preprocessedSets = preprocessMask(preprocessList);
    } else if (preprocessMode == PreprocessMode.FILTER) {
      conductor.message("sequence sets before filtering: " + sequenceSets.size());
      preprocessedSets = preprocessFilter(preprocessList);
      conductor.message("sequence sets after filtering: " + preprocessedSets.size());
      totalWeight = countSequences(preprocessedSets);
    }
    if (preprocessedSets.size() == 0) return;

    List<Double> setWeights = new ArrayList<Double>();
    if (logWeighting) {
      conductor.message("using logarithmic sets weighting");
      double sumW = 0.0;
      for (int i = 0; i < preprocessedSets.size(); i++) {
        Sequence[] preprocessedSet = preprocessedSets.get(i);
        setWeights.add(Math.log(preprocessedSet.length + 1));
        sumW += setWeights.get(i);
      }
      double koeff = totalWeight / sumW;
      for (int i = 0; i < preprocessedSets.size(); i++)
        setWeights.set(i, setWeights.get(i)*koeff);

    } else {
      double effSize = 0;
      for (int i = 0; i < preprocessedSets.size(); i++) effSize += preprocessedSets.get(i).length > 0 ? 1.0 : 0.0;
      for (int i = 0; i < preprocessedSets.size(); i++)
        setWeights.add(totalWeight / effSize);
    }

    for (int i = 0; i < preprocessedSets.size(); i++) {
      Sequence[] sequenceSet = preprocessedSets.get(i);
      normalizeWeights(sequenceSet, setWeights.get(i));
    }
    joinedSets = joinSeqsa(totalWeight, preprocessedSets);

    conductor.message("using " + joinedSets.length + " sequences");

  }

  public ChIPAct(Parameters parameters) {
    super(parameters.getConductor());

    conductor.setTotalTicks(9);

    this.motifLength = parameters.getMotifLength();
    if (motifLength < 1) throw new RuntimeException("too small motif length");

    this.threadCount = parameters.getThreadCount();
    this.iterationLimit = parameters.getIterationLimit();
    this.stepLimit = parameters.getStepLimit();
    this.tryLimit = parameters.getTryLimit();
    this.shape = parameters.getShape();

    if (parameters.isLocalBackground()) {
      background = Sequence.background(joinedSets);
    }

    this.sequenceSets = parameters.getSequenceSets();
    this.prepareSets(parameters.isLogWeighting());

    makeJoinedSets(parameters.getPreprocessMode(), parameters.getPreprocessList(), parameters.isLogWeighting());
    if (joinedSets == null || joinedSets.length == 0) throw new RuntimeException("empty sequence set: everything is filtered out or sequences are shorter than the requested motif length");

    conductor.message("gathering seeds...");
    seeds = new SeedCask(tryLimit, motifLength);
    conductor.message("gathered " + seeds.count() + " seeds");

  }

  private void prepareSets(boolean logWeighting) {

    List<Sequence[]> result = new ArrayList<Sequence[]>(sequenceSets.size());
    for (int i = 0; i < sequenceSets.size(); i++) {
      List<Sequence> filtered = new ArrayList<Sequence>();
      Sequence[] set = sequenceSets.get(i);
      for (Sequence seq: set) {

        if (seq.weight <= 0.0)
          throw new RuntimeException("found a sequence with zero weight");
        else if (seq.getDirect().length != seq.getRevcomp().length)
          throw new RuntimeException("internal error: found a sequence with different direct and reverse complementary strand lengths");
        else if (seq.getDirect().length < 2)
          throw new RuntimeException("found too short sequence of length " + seq.getDirect().length);

        if (seq.getDirect().length >= motifLength)
          filtered.add(seq.copy());
          // safe way: duplicate initial data, so profiles/weights are not damaged when ChIPAct finishes
      }
      Sequence[] resf = filtered.toArray(new Sequence[filtered.size()]);

      if (resf.length > 0) result.add(resf);
    }
    if (result.size() == 0) throw new RuntimeException("found no suitable sequences for the selected motif length " + motifLength);
    sequenceSets = result;

    for (Sequence[] seqs: sequenceSets) {
      for (Sequence seq: seqs) {
        if (seq.getClass() == Peak.class) {
          Peak peak = (Peak)seq;
          peak.soften(motifLength);
          peak.normalize(logWeighting);
        }
      }
    }

  }

  public WPCM launch() {
    if (joinedSets == null) throw new RuntimeException("no sequence data present (possibly everything filtered out)");

    ExecutorService threadPool = java.util.concurrent.Executors.newFixedThreadPool(threadCount, conductor.getThreadFactory());

    WPCM seed;
    Future lastTaskRes = null; // for debugging need to be modified to save results of all tasks 

    double complete = 0.1;
    while ((seed = seeds.seed()) != null) {
      Task task = new Task(new DoptiStep(joinedSets, Math.sqrt(joinedSets.length), seed, this, stepLimit, iterationLimit, background, shape), this);
      double x = 1.0 - (seeds.size() + 1.0) / tryLimit;
      if ( x >= complete ) {
        complete += 0.1;
        task.setMessage("motif discovery, length " + motifLength + ": " + java.lang.Math.floor(java.lang.Math.round(x * 10.0) * 10.0) + "% complete");
      }
      lastTaskRes = threadPool.submit(task);
    }

    threadPool.shutdown();
    try {
      threadPool.awaitTermination(Long.MAX_VALUE, TimeUnit.DAYS);
    } catch (Exception e) {
      // conductor.printStackTrace(e);
      throw new RuntimeException(e);
    }

    conductor.output("OUTC", "ru.autosome.di.ChIPAct");
    if (best != null) {
      conductor.message("success; best KDIDIC " + best.kdidic(background));
      conductor.output("DIAG", "gapless multiple local alignment of length " + best.length());
      conductor.output("KDDC", best.kdidic(background));
      if (verbose) prettyOutputWPCM(conductor, best.getMatrix());
    } else {
      try {
        lastTaskRes.get();
      } catch (Exception e) {
        // conductor.printStackTrace(e);
        throw new RuntimeException(e);
      }
    }

    result = best;
    conductor.setStatus(result != null ? Conductor.Status.SUCCESS : Conductor.Status.FAIL);

    return result;
  }

  public static void normalizeWeights(Sequence[] sequenceSet, double v) {

    double weightSum = 0;
    for (Sequence sequence : sequenceSet) {
      if (sequence.weight <= 0.0) throw new RuntimeException("found sequence with the zero weight or peak height");
      weightSum += sequence.weight;
    }
    double koeff = v / weightSum;
    for (Sequence sequence : sequenceSet) {
      sequence.weight *= koeff;
    }

  }

  public static Sequence[] loadSequences(String filename, int minLength) {
    List<Sequence> resa = new ArrayList<Sequence>();

    try {
      BufferedReader bin = new BufferedReader(new InputStreamReader(new FileInputStream(filename)));
      String s;
      StringBuilder sb = new StringBuilder();

      while ((s = bin.readLine()) != null) {
        if (s.length() > 0 && s.charAt(0) == '>') {
          if (sb.length() >= minLength) {
            resa.add(new Sequence(sb.toString()));
          }
          sb.setLength(0);
          continue;
        }
        sb.append(s.trim());
      }
      if (sb.length() >= minLength) resa.add(new Sequence(sb.toString()));

    } catch (IOException e) {
      throw new RuntimeException("unable to parse sequences file " + filename);
    }

    Sequence[] res = new Sequence[resa.size()];
    return resa.toArray(res);
  }

  public static Sequence[] loadWeighted(String filename, int minLength) {
    List<Sequence> resa = new ArrayList<Sequence>();

    try {
      BufferedReader bin = new BufferedReader(new InputStreamReader(new FileInputStream(filename)));
      String s;
      StringBuilder sb = new StringBuilder();
      double weight = 0.0;

      while ((s = bin.readLine()) != null) {
        if (s.length() > 0 && s.charAt(0) == '>') {
          // load weight
          if (sb.length() >= minLength) {
            resa.add(new Sequence(sb.toString(), weight));
          }
          sb.setLength(0);
          weight = Double.parseDouble(s.substring(1).trim().split(" ")[0]);
          continue;
        }
        sb.append(s.trim());
      }
      if (sb.length() >= minLength) resa.add(new Sequence(sb.toString(), weight));

    } catch (IOException e) {
      throw new RuntimeException("unable to parse sequences file " + filename);
    }

    Sequence[] res = new Sequence[resa.size()];
    return resa.toArray(res);
  }

  public static Peak[] loadPeaks(String filename, int minLength) {
    List<Peak> resa = new ArrayList<Peak>();

    try {
      BufferedReader bin = new BufferedReader(new InputStreamReader(new FileInputStream(filename)));
      String s;
      StringBuilder sb = new StringBuilder();
      double[] profile = null;

      while ((s = bin.readLine()) != null) {
        if (s.length() != 0 && s.charAt(0) == '>') {
          String[] pivis = s.substring(1).trim().split(" ");
          if (sb.length() >= minLength) {
            if (profile.length != sb.length()) {
              throw new IOException("the profile length is not equal to the sequence length in the multifasta");
            }
            resa.add(new Peak(sb.toString(), profile));
          }
          sb.setLength(0);
          profile = new double[pivis.length];
          for (int i = 0; i < pivis.length; i++) {
            profile[i] = Double.parseDouble(pivis[i]);
          }
          continue;
        }
        sb.append(s.trim());
      }
      if (sb.length() >= minLength) {
        if (profile.length != sb.length()) {
          throw new IOException("the profile length is not equal to the sequence length in the multifasta");
        }
        resa.add(new Peak(sb.toString(), profile));
      }

    } catch (IOException e) {
      e.printStackTrace();
      throw new RuntimeException("unable to parse the sequences file " + filename);
    }

    Peak[] res = new Peak[resa.size()];

    return resa.toArray(res);
  }

  public static Sequence[] joinSeqsa(int size, List<Sequence[]> sequences) {

    // create list of appropriate size
    List<Sequence> list = new ArrayList<Sequence>(size);

    // add arrays
    for (Sequence[] array : sequences) {
      list.addAll(java.util.Arrays.asList(array));
    }

    // create and return final array
    return list.toArray(new Sequence[size]);
  }

  private static int countSequences(List<Sequence[]> preprocessedSets) {
    int count = 0;
    for (Sequence[] array: preprocessedSets) {
      count += array.length;
    }
    return count;
  }

  private List<Sequence[]> preprocessFilter(List<MunkResult> preprocessList) {
    List<Sequence[]> result = new ArrayList<Sequence[]>(sequenceSets.size());

    for (Sequence[] sequenceSet : sequenceSets) {
      List<Sequence> nonfiltered = new ArrayList<Sequence>(sequenceSet.length);
      for (Sequence s: sequenceSet) {
        boolean filter = false;
        for (int i = 0, preprocessListSize = preprocessList.size(); i < preprocessListSize; i++) {
          WPCM pm = preprocessList.get(i).getPWM();
          double threshold = preprocessList.get(i).getThreshold();
          filter = s.hasOccurrence(pm, threshold);
          if (filter) break;
        }
        if (!filter) nonfiltered.add(s);

      }
      if (nonfiltered.size() > 0) result.add(nonfiltered.toArray(new Sequence[nonfiltered.size()]));
    }

    return result;
  }

  private List<Sequence[]> preprocessMask(List<MunkResult> preprocessList) {
    List<Sequence[]> result = new ArrayList<Sequence[]>(sequenceSets.size());

    for (Sequence[] sequenceSet : sequenceSets) {
      List<Sequence> maskedSet = new ArrayList<Sequence>(sequenceSet.length);
      for (Sequence s: sequenceSet) {
        Sequence masked = s.copy();

        // TODO: beautify RNASequence usage - i.e. when there is no revcomp strand is present
        Integer rclength = masked.getRevcomp() == null ? null : masked.getRevcomp().length;

        for (int i = 0, preprocessListSize = preprocessList.size(); i < preprocessListSize; i++) {
          WPCM pm = preprocessList.get(i).getPWM();
          double threshold = preprocessList.get(i).getThreshold();

          for (Occurrence occ : s.gatherOccurrences(pm, threshold)) {
            int position = occ.getPosition();
            for (int j = position; j < position + pm.length(); j++) {
              masked.getDirect()[j] = (byte)Din.NN.ordinal();
              masked.getRevcomp()[rclength - j - 1] = (byte)Din.NN.ordinal();
            }
          }
        }

        maskedSet.add(masked);

      }
      result.add(maskedSet.toArray(new Sequence[maskedSet.size()]));
    }

    return result;
  }

  public static  void prettyOutputWPCM(Conductor conductor, double[][] matrix) {
    double weight = 0.0;
    for (byte i = 0; i <= 15; i++) {
      String s = Conductor.prettyString(matrix[i]);
      conductor.output(Din.map.get(i), s);
      weight += matrix[i][0];
    }
    conductor.output("NN", weight);
  }

  public static void prettyOutputPWM(Conductor conductor, double[][] matrix) {
    for (byte i = 0; i <= 15; i++) {
      String s = Conductor.prettyString(matrix[i]);
      conductor.output("PW"+Din.map.get(i), s);
    }
  }

  public static void prettyOutputBackground(Conductor conductor, double[] background) {
    conductor.output("ORDR", "AA AC AG AT CA CC CG CT GA GC GG GT TA TC TG TT");
    StringBuilder baks = new StringBuilder();
    baks.append(background[Din.AA.ordinal()]).append(" ");
    baks.append(background[Din.AC.ordinal()]).append(" ");
    baks.append(background[Din.AG.ordinal()]).append(" ");
    baks.append(background[Din.AT.ordinal()]).append(" ");
    baks.append(background[Din.CA.ordinal()]).append(" ");
    baks.append(background[Din.CC.ordinal()]).append(" ");
    baks.append(background[Din.CG.ordinal()]).append(" ");
    baks.append(background[Din.CT.ordinal()]).append(" ");
    baks.append(background[Din.GA.ordinal()]).append(" ");
    baks.append(background[Din.GC.ordinal()]).append(" ");
    baks.append(background[Din.GG.ordinal()]).append(" ");
    baks.append(background[Din.GT.ordinal()]).append(" ");
    baks.append(background[Din.TA.ordinal()]).append(" ");
    baks.append(background[Din.TC.ordinal()]).append(" ");
    baks.append(background[Din.TG.ordinal()]).append(" ");
    baks.append(background[Din.TT.ordinal()]);
    conductor.output("BACK", baks.toString());
  }

  public static class Parameters {
    private Conductor conductor;
    private PreprocessMode preprocessMode;
    private List<MunkResult> preprocessList;
    private List<Sequence[]> sequenceSets;
    private Integer motifLength;
    private int threadCount;
    private int iterationLimit;

    private int stepLimit;
    private int tryLimit;
    private boolean logWeighting;
    private boolean localBackground;
    private double[] shape;

    public Parameters(Conductor conductor, PreprocessMode preprocessMode, List<MunkResult> preprocessList, List<Sequence[]> sequenceSets, int motifLength, int tryLimit, int stepLimit, int iterationLimit, int threadCount, boolean localBackground, boolean logWeighting) {
      this.conductor = conductor;
      this.preprocessMode = preprocessMode;
      this.preprocessList = preprocessList;
      this.sequenceSets = sequenceSets;
      this.motifLength = motifLength;
      this.threadCount = threadCount;
      this.iterationLimit = iterationLimit;
      this.stepLimit = stepLimit;
      this.tryLimit = tryLimit;
      this.logWeighting = logWeighting;
      this.localBackground = localBackground;
    }

    public Parameters(Conductor conductor, List<Sequence[]> sequenceSets, Integer motifLength) {
      this.conductor = conductor;
      this.preprocessMode = null;
      this.preprocessList = null;
      this.sequenceSets = sequenceSets;
      this.motifLength = motifLength;
      this.threadCount = 2;
      this.iterationLimit = 1;
      this.stepLimit = 20;
      this.tryLimit = 200;
      this.logWeighting = true;
      this.localBackground = false;
    }

    public Parameters(Conductor conductor, List<Sequence[]> sequenceSets) {
      this(conductor, sequenceSets, null);
    }

    public Conductor getConductor() {
      return conductor;
    }

    public PreprocessMode getPreprocessMode() {
      return preprocessMode;
    }

    public List<MunkResult> getPreprocessList() {
      return preprocessList;
    }

    public List<Sequence[]> getSequenceSets() {
      return sequenceSets;
    }

    public int getMotifLength() {
      return motifLength;
    }

    public int getThreadCount() {
      return threadCount;
    }

    public int getIterationLimit() {
      return iterationLimit;
    }

    public int getStepLimit() {
      return stepLimit;
    }

    public int getTryLimit() {
      return tryLimit;
    }

    public boolean isLogWeighting() {
      return logWeighting;
    }

    public boolean isLocalBackground() {
      return localBackground;
    }

    public double[] getShape() {
      return shape;
    }

    public void setPreprocessMode(PreprocessMode preprocessMode) {
      this.preprocessMode = preprocessMode;
    }

    public void setPreprocessList(List<MunkResult> preprocessList) {
      this.preprocessList = preprocessList;
    }

    public void setThreadCount(int threadCount) {
      this.threadCount = threadCount;
    }

    public void setIterationLimit(int iterationLimit) {
      this.iterationLimit = iterationLimit;
    }

    public void setStepLimit(int stepLimit) {
      this.stepLimit = stepLimit;
    }

    public void setTryLimit(int tryLimit) {
      this.tryLimit = tryLimit;
    }

    public void setLogWeighting(boolean logWeighting) {
      this.logWeighting = logWeighting;
    }

    public void setLocalBackground(boolean localBackground) {
      this.localBackground = localBackground;
    }

    public void setMotifLength(int motifLength) {
      this.motifLength = motifLength;
    }

    public void setShape(double[] shape) {
      this.shape = shape;
    }

  }
}
