package ru.autosome.assist;

import ru.autosome.ChIPApp;

import java.util.concurrent.ThreadFactory;

public class Conductor {
  private static final String VERSION = "V8 02022020";

  public void setOutputPrinter(IPrinter outputPrinter) {
    this.outputPrinter = outputPrinter;
  }
  public void setMessagePrinter(IPrinter messagePrinter) {
    this.messagePrinter = messagePrinter;
  }
  public Exception getError() {
    return error;
  }
  public void setThreadFactory(ThreadFactory tf) {
    threadFactory = tf;
  }
  public void interrupt() {
    synchronized (status) {
      status = Status.INTERRUPTED;
    }
  }

  private ThreadFactory threadFactory = java.util.concurrent.Executors.defaultThreadFactory();
  public ThreadFactory getThreadFactory() {
    return threadFactory;
  }

  private final long TIME = System.currentTimeMillis();
  private Status status = Status.INITIALIZED;

  private IPrinter outputPrinter = new IPrinter() {
    public void println(String message) {
      System.out.println(message);
    }
  };

  private IPrinter messagePrinter = new IPrinter() {
    public void println(String message) {
      System.err.println(message);
    }
  };

  private Exception error = null;
  public static Conductor defaultConductor = new Conductor();

  private boolean silence = false;

  public void output(Object key, Object value) {
    synchronized(status) {
      if (status == Status.INTERRUPTED) throw new RuntimeException("execution interrupted");
      if (!silence) outputPrinter.println(key.toString() + "|" + value.toString());
    }
  }

  public void message(String message) {
    synchronized (status) {
      if (status == Status.INTERRUPTED) throw new RuntimeException("execution interrupted");
      messagePrinter.println(message);
    }
  }

  public Object launch(ChIPApp app) {
    Object result;
    output("PROG", app.getClass().getCanonicalName() + " " + Conductor.VERSION);
    try {
      status = Status.RUNNING;
      result = app.launch();
      status = Status.WAITING;
    } catch (Exception e) {
      String message = e.toString() == null ? "no information" : e.toString();
      if (status == Status.INTERRUPTED) {
        synchronized (status) {
          messagePrinter.println("interrupted (" + app.getClass().getCanonicalName() + "): " + message);
          if (!silence) outputPrinter.println("INTR|" + app.getClass().getCanonicalName());
        }
        return null;
      }
      else {
        status = Status.ERROR;
        messagePrinter.println("error: (" + app.getClass().getCanonicalName() + "): " + message);

        // used for DEBUG
        e.printStackTrace();

        if (!silence) outputPrinter.println("ERRR|" + message);
        error = e;
        return null;
      }
    }
    output("TIME", (System.currentTimeMillis() - TIME) / 1000.0);
    output("DUMP", app.getClass().getCanonicalName() + " " + Conductor.VERSION);
    output("/\\/\\", " -------------------------- ");
    output("_^^_", " P0wered by cute chipmunks! ");

    return result;
  }

  public void silence() {
    silence = true;
  }

  public void verbose() {
    silence = false;
  }

  private Integer totalTicks = null;
  private Integer currentTicks = null;

  public Integer getDone() {
    return done;
  }

  private Integer done = 0;
  public void setTotalTicks(Integer totalTicks) {
    if (this.totalTicks == null) {
      this.totalTicks = totalTicks;
      this.currentTicks = 0;
    }
  }
  public synchronized void tick(String message) {
    currentTicks += 1;
    done = Math.max(done, (int)(currentTicks*100.0/totalTicks)-1);

    int remaining = 100 - done;
    double currentTimeSec = (System.currentTimeMillis() - TIME) / 1000.0;
    double speed = done / currentTimeSec;
    double remainingTimeSec = remaining * speed;

    // if (done > 0) this.message("overall progress: " + done + "% complete (" + message + "); remaming time approx. " + ((int)(remainingTimeSec / 60)) + " min.") ;
    if (done > 0) this.message("overall progress: " + done + "% complete (" + message + ")") ;
  }
  public void updateTicks(int ticks) {
    if (currentTicks != ticks) {
      currentTicks = ticks;
      done = Math.max(done, (int)(currentTicks*100.0/totalTicks)-1);
      this.message("overall progress: " + done + "% complete");
    }
  }

  public static String prettyString(double[] ds) {
    StringBuilder sb = new StringBuilder(Double.toString(ds[0]));
    for (int i = 1; i < ds.length; i++) {
      sb.append(" ").append(ds[i]);
    }
    return sb.toString();
  }

  public void check() {
    synchronized (status) {
      if (status == Status.INTERRUPTED) throw new RuntimeException("execution interrupted");
    }
  }

  public Status getStatus() {
    return status;
  }

  public void setStatus(Status newStatus) {
    synchronized (status) {
      if (newStatus != status && status == Status.INTERRUPTED) throw new RuntimeException("execution interrupted");
      status = newStatus;
    }
  }

  public static enum Status {
    INITIALIZED, RUNNING, WAITING, INTERRUPTED, SUCCESS, FAIL, ERROR
  }
}
