package ru.autosome;

import ru.autosome.assist.ASequence;
import ru.autosome.assist.AShapeProvider;
import ru.autosome.assist.Conductor;
import ru.autosome.engine.SoptiStep;
import ru.autosome.ytilib.MunkResult;
import ru.autosome.ytilib.WPCM;

import java.util.List;

public class ChIPMunk extends ChIPApp {
  private PreprocessMode actMode = null;
  private List<MunkResult> actPreprocessingList = null;

  int minLength;
  int maxLength;

  Double zoopsFactor;
  boolean motifFound;
  Integer motifLength;
  int delta;
  double[] background = null;
  ChIPAct.Parameters actParameters;
  AShapeProvider shapeProvider = null;

  MunkResult result = null;
  public MunkResult getResult() {
    return result;
  }

  public static void main(String[] args) {
    ChIPMunk chipmunk = new ChIPMunk(args, Conductor.defaultConductor);
    chipmunk.launchViaConductor();
  }

  public ChIPMunk(ChIPAct.Parameters actParameters, ChIPMunk.Parameters munkParameters) {
    super(actParameters.getConductor());

    minLength = munkParameters.getStartLength();
    maxLength = munkParameters.getStopLength();
    conductor.setTotalTicks((Math.abs(maxLength - minLength) + 1) * 9);

    delta = 1;
    if (maxLength < minLength) {
      delta = -1;
      minLength = munkParameters.getStopLength();
      maxLength = munkParameters.getStartLength();
    }

    verbose = munkParameters.getVerbose();
    zoopsFactor = munkParameters.getZoopsFactor();

    this.actParameters = actParameters;
    this.shapeProvider = munkParameters.getShapeProvider();

    motifFound = false;
    motifLength = delta > 0 ? minLength : maxLength;
  }

  String[] params;

  public ChIPMunk(String[] params, Conductor conductor, ChIPAct.PreprocessMode mode, List<MunkResult> preprocessingList) {
    this(params, conductor);
    this.actMode = mode;
    this.actPreprocessingList = preprocessingList;
  }

  public ChIPMunk(String[] args, Conductor conductor) {
    super(conductor);

    conductor.message("ru.autosome.ChIPMunk usage: <start_length> <stop_length> <verbose=(n)o|(y)es> <mode=(o)ops|zoops_factor=1.0|0.0> <ru.autosome.ChIPAct parameters>");
    if (args.length == 1) {
      // DEFAULT MODE, assuming sequence set is given
      verbose = true;
      minLength = 7;
      maxLength = 22;
      conductor.setTotalTicks((Math.abs(maxLength - minLength) + 1) * 9);

      zoopsFactor = 1.0;

      motifFound = false;
      motifLength = 7;
      delta = 1;

      params = new String[]{null, args[0], "100", "10", "1", "2"};

    } else if (args.length >= 5) {

      minLength = new Integer(args[0]);
      maxLength = new Integer(args[1]);
      conductor.setTotalTicks((Math.abs(maxLength - minLength) + 1) * 9);

      delta = 1;
      if (maxLength < minLength) {
        delta = -1;
        minLength = new Integer(args[1]);
        maxLength = new Integer(args[0]);
      }

      verbose = args[2].toUpperCase().charAt(0) == 'Y';

      boolean oops = args[3].toUpperCase().charAt(0) == 'O';
      zoopsFactor = null;
      if (!oops) zoopsFactor = new Double(args[3]);

      params = new String[args.length - 3];
      System.arraycopy(args, 4, params, 1, params.length - 1);

      motifFound = false;
      motifLength = delta > 0 ? minLength : maxLength;

    } else throw new RuntimeException("not enough parameters");
  }

  public MunkResult launch() {

    MunkResult newResult = null;
    MunkResult result = null;

    while (motifLength <= maxLength && motifLength >= minLength && !motifFound) {

      conductor.message("trying motif length: " + motifLength);

      ChIPAct step;

      if (params != null) {
        params[0] = motifLength.toString();
        step = new ChIPAct(params, conductor, actMode, actPreprocessingList);
      } else {
        actParameters.setMotifLength(motifLength);
        if (shapeProvider != null) actParameters.setShape(shapeProvider.shape(motifLength));
        step = new ChIPAct(actParameters);
      }
      step.launchViaConductor();

      if (step.getResult() == null) {
        conductor.message("no motif found, please check your input data and parameters");
        return null;
      }
      if (background == null) background = step.background;

      boolean badNewMotif;

      WPCM respcm = step.getBest();
      conductor.message("checking motif strength...");
      SoptiStep basestep = new SoptiStep(step.joinedSets, step.joinedSets.length, background, SoptiStep.GLOBAL_ITERATION_LIMIT);
      basestep.optimize(respcm);

      if (zoopsFactor == null) {

        WPCM reswoweights = new WPCM(motifLength, basestep.getPrimaryHits(), basestep.getRevcompHits(), step.joinedSets, true);

        badNewMotif = !checkMotif(respcm, background);
        badNewMotif = badNewMotif || !checkMotif(reswoweights, background);

        newResult = new MunkResult(basestep.extractWordList(new WPCM(respcm).toPWM(background)), respcm, background, step.joinedSets.length, step.joinedSets.length, ASequence.medianLength(step.joinedSets));

      } else {

        conductor.message("applying ZOOPS segmentation procedure...");

        newResult = basestep.zoops(zoopsFactor, conductor);
        badNewMotif = !checkMotif(newResult.getWPCM(), background);
      }

      if (delta < 0) {
        motifFound = !badNewMotif;
        if (motifFound) {
          conductor.message("motif strength check successfully passed, taking current motif");
          result = newResult;
        }
      } else {
        motifFound = false;
        if (result != null && badNewMotif) {
          conductor.message("motif strength check not passed, taking previously found motif");
          motifLength = motifLength - 1;
          motifFound = true;
        } else if (result == null && badNewMotif) {
          conductor.message("detected weak starting motif");
          break;
        } else if (!badNewMotif) {
          conductor.message("found strong motif, saving");
          result = newResult;
        }
      }

      if (!motifFound) motifLength += delta;

    }

    conductor.output("OUTC", "ru.autosome.ChIPMunk");
    if (result != null) {
      if (motifLength >= maxLength)
        conductor.message("found motif of the maximal allowed length " + maxLength + "; please, check input parameters");
      if (motifLength <= minLength)
        conductor.message("found motif of the minimal allowed length " + minLength + "; please, check input parameters");

      conductor.setStatus(Conductor.Status.SUCCESS);

      if (motifFound) conductor.message("success; final motif length: " + motifLength);
      result.printout(conductor, verbose);

      conductor.output("LENG", motifFound ? motifLength : -1);
      conductor.output("KDIC", result.getWPCM().kdic(background));
      ChIPAct.prettyOutputWPCM(conductor, result.getWPCM().getMatrix());
      ChIPAct.prettyOutputPWM(conductor, result.getPWM().getMatrix());
      conductor.output("THRE", result.getThreshold());
      conductor.output("IUPA", result.getWPCM().consensus1());

      if (zoopsFactor != null) {
        conductor.message("estimating P-value...");
        conductor.output("PVAL", result.pvalue());
      }
      ChIPAct.prettyOutputBackground(conductor, background);

      result.setDiagnosis(motifFound ? "success; strong motif found within the given lengths interval" : "partial success; strong motif found on the border of the given lengths interval");
      conductor.output("DIAG", result.getDiagnosis());
    } else {

      conductor.setStatus(Conductor.Status.FAIL);
      conductor.message("only weak motifs found; please extend allowed lengths range");

      newResult.printout(conductor, verbose);

      conductor.output("LENG", -1);
      conductor.output("KDIC", newResult.getWPCM().kdic(background));
      ChIPAct.prettyOutputWPCM(conductor, newResult.getWPCM().getMatrix());
      ChIPAct.prettyOutputPWM(conductor, newResult.getPWM().getMatrix());
      conductor.output("THRE", newResult.getThreshold());
      conductor.output("IUPA", newResult.getWPCM().consensus1());

      if (zoopsFactor != null) {
        conductor.message("estimating P-value...");
        conductor.output("PVAL", newResult.pvalue());
      }
      ChIPAct.prettyOutputBackground(conductor, background);

      newResult.setDiagnosis("fail; strong motif not found within the given lengths range");
      conductor.output("DIAG", newResult.getDiagnosis());
    }

    return result != null ? result : newResult;
  }

  private static boolean checkMotif(WPCM mocheck, double[] background) {

    if (mocheck == null) return false;

    double leftborder = mocheck.kdic(background, 0), rightborder = mocheck.kdic(background, mocheck.length() - 1);
    double icdLow = mocheck.thresholdLC(), icdHigh = mocheck.thresholdHC();

    if (leftborder < icdLow || rightborder < icdLow) return false;

    double[] kdic = new double[mocheck.length()];
    for (int i = 0; i < kdic.length; i++) {
      kdic[i] = mocheck.kdic(background, i);
    }

    char[] motif = new char[mocheck.length()];
    for (int i = 0; i < motif.length; i++) {
      motif[i] = kdic[i] < icdLow ? 'N' : (kdic[i] < icdHigh ? 's' : 'B');
    }

    String motifS = new String(motif);
    String[] motifP = motifS.split("N");
    for (String s : motifP) {
      if (s.length() == 0) continue;
      if (s.indexOf("B") < 0) return false;
    }

    return true;
  }


  public static class Parameters {

    public Parameters() {
      verbose = true;
      zoopsFactor = 1.0;
    }

    public Parameters(int startLength, int stopLength, boolean verbose, Double zoopsFactor) {
      this.verbose = verbose;
      this.zoopsFactor = zoopsFactor;
      this.startLength = startLength;
      this.stopLength = stopLength;
    }

    private int startLength;
    private int stopLength;
    private boolean verbose;
    private Double zoopsFactor;

    public void setShapeProvider(AShapeProvider shapeProvider) {
      this.shapeProvider = shapeProvider;
    }

    private AShapeProvider shapeProvider;

    public int getStartLength() {
      return startLength;
    }

    public int getStopLength() {
      return stopLength;
    }

    public Boolean getVerbose() {
      return verbose;
    }

    public Double getZoopsFactor() {
      return zoopsFactor;
    }

    public void setStartLength(int startLength) {
      this.startLength = startLength;
    }

    public void setStopLength(int stopLength) {
      this.stopLength = stopLength;
    }

    public void setVerbose(Boolean verbose) {
      this.verbose = verbose;
    }

    public void setZoopsFactor(Double zoopsFactor) {
      this.zoopsFactor = zoopsFactor;
    }

    public AShapeProvider getShapeProvider() {
      return shapeProvider;
    }
  }
}